import { useState } from 'react';
import { useTranslation } from 'react-i18next';
import { Send, Facebook, Twitter, Linkedin, Instagram, Youtube, ArrowUp } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useContent } from '@/context/ContentContext';

export default function Footer() {
  const { t } = useTranslation();
  const { content } = useContent();
  const [email, setEmail] = useState('');
  const [subscribed, setSubscribed] = useState(false);

  const handleSubscribe = (e: React.FormEvent) => {
    e.preventDefault();
    if (email) {
      setSubscribed(true);
      setEmail('');
    }
  };

  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const socialLinks = [
    { icon: Facebook, href: content.contact.socialLinks.facebook, label: 'Facebook' },
    { icon: Twitter, href: content.contact.socialLinks.twitter, label: 'Twitter' },
    { icon: Linkedin, href: content.contact.socialLinks.linkedin, label: 'LinkedIn' },
    { icon: Instagram, href: content.contact.socialLinks.instagram, label: 'Instagram' },
    { icon: Youtube, href: content.contact.socialLinks.youtube, label: 'YouTube' }
  ];

  return (
    <footer className="bg-[#050508] border-t border-white/5">
      <div className="container-custom py-16">
        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-12">
          {/* Company Info */}
          <div className="lg:col-span-1">
            <a href="#hero" className="flex items-center gap-2 mb-6">
              <div className="w-10 h-10 rounded-lg bg-gradient-to-br from-red-500 to-orange-500 flex items-center justify-center">
                <span className="text-white font-bold text-lg">Z</span>
              </div>
              <span className="text-white font-bold text-xl">
                ZLO <span className="text-red-500">Technologies</span>
              </span>
            </a>
            <p className="text-white/50 text-sm leading-relaxed mb-6">
              {content.footer.companyDescription}
            </p>
            <div className="flex gap-3">
              {socialLinks.map((social, index) => (
                social.href && (
                  <a
                    key={index}
                    href={social.href}
                    aria-label={social.label}
                    className="w-9 h-9 rounded-full bg-white/5 flex items-center justify-center hover:bg-red-500/20 hover:text-red-400 transition-colors"
                  >
                    <social.icon className="w-4 h-4 text-white/60" />
                  </a>
                )
              ))}
            </div>
          </div>

          {/* Quick Links */}
          <div>
            <h4 className="text-white font-semibold mb-6">{t('footer.quickLinks')}</h4>
            <ul className="space-y-3">
              {content.footer.quickLinks.map((link, index) => (
                <li key={index}>
                  <a
                    href={link.href}
                    className="text-white/50 hover:text-red-400 transition-colors text-sm"
                  >
                    {link.label}
                  </a>
                </li>
              ))}
            </ul>
          </div>

          {/* Services */}
          <div>
            <h4 className="text-white font-semibold mb-6">{t('footer.services')}</h4>
            <ul className="space-y-3">
              {content.footer.services.map((service, index) => (
                <li key={index}>
                  <a
                    href={service.href}
                    className="text-white/50 hover:text-red-400 transition-colors text-sm"
                  >
                    {service.label}
                  </a>
                </li>
              ))}
            </ul>
          </div>

          {/* Newsletter */}
          <div>
            <h4 className="text-white font-semibold mb-6">{t('footer.newsletter')}</h4>
            <p className="text-white/50 text-sm mb-4">
              {content.footer.newsletterDescription}
            </p>
            {subscribed ? (
              <p className="text-green-400 text-sm">Merci pour votre inscription !</p>
            ) : (
              <form onSubmit={handleSubscribe} className="flex gap-2">
                <Input
                  type="email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  placeholder="votre@email.com"
                  className="bg-white/5 border-white/10 text-white placeholder:text-white/30 focus:border-red-500 text-sm"
                  required
                />
                <Button
                  type="submit"
                  size="icon"
                  className="bg-gradient-to-r from-red-500 to-orange-500 hover:from-red-600 hover:to-orange-600 shrink-0"
                >
                  <Send className="w-4 h-4" />
                </Button>
              </form>
            )}
          </div>
        </div>

        {/* Bottom Bar */}
        <div className="mt-16 pt-8 border-t border-white/5 flex flex-col sm:flex-row justify-between items-center gap-4">
          <p className="text-white/40 text-sm">
            {content.footer.copyright}
          </p>
          <div className="flex gap-6">
            {content.footer.legalLinks.map((link, index) => (
              <a
                key={index}
                href={link.href}
                className="text-white/40 hover:text-white/60 transition-colors text-sm"
              >
                {link.label}
              </a>
            ))}
          </div>
        </div>
      </div>

      {/* Scroll to top button */}
      <button
        onClick={scrollToTop}
        className="fixed bottom-8 right-8 w-12 h-12 rounded-full bg-gradient-to-r from-red-500 to-orange-500 flex items-center justify-center shadow-lg shadow-red-500/30 hover:shadow-red-500/50 transition-shadow z-40"
        aria-label="Scroll to top"
      >
        <ArrowUp className="w-5 h-5 text-white" />
      </button>
    </footer>
  );
}
