import { useEffect, useRef } from 'react';
import { ArrowRight, ChevronDown } from 'lucide-react';
import gsap from 'gsap';

interface HeroProps {
  onNavigate?: (section: string) => void;
}

export default function Hero({ onNavigate }: HeroProps) {
  const heroRef = useRef<HTMLDivElement>(null);
  const titleRef = useRef<HTMLDivElement>(null);
  const imageRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Animate title lines
      gsap.fromTo(
        '.hero-title-line',
        { y: 80, opacity: 0 },
        { y: 0, opacity: 1, duration: 0.8, stagger: 0.1, ease: 'power3.out', delay: 0.3 }
      );

      // Animate subtitle
      gsap.fromTo(
        '.hero-subtitle',
        { y: 30, opacity: 0 },
        { y: 0, opacity: 1, duration: 0.6, ease: 'power3.out', delay: 0.8 }
      );

      // Animate CTA
      gsap.fromTo(
        '.hero-cta',
        { y: 30, opacity: 0 },
        { y: 0, opacity: 1, duration: 0.6, ease: 'power3.out', delay: 1 }
      );

      // Animate image
      gsap.fromTo(
        imageRef.current,
        { scale: 0.9, opacity: 0 },
        { scale: 1, opacity: 1, duration: 1, ease: 'power3.out', delay: 0.5 }
      );
    }, heroRef);

    return () => ctx.revert();
  }, []);

  const handleNavClick = (href: string) => {
    if (onNavigate) {
      onNavigate(href);
    }
  };

  return (
    <section
      ref={heroRef}
      id="hero"
      className="relative min-h-screen flex items-center overflow-hidden bg-[#0a0a0a]"
    >
      {/* Grid lines background */}
      <div className="absolute inset-0 grid-lines opacity-50" />

      {/* Content */}
      <div className="container-custom relative z-10 pt-20">
        <div className="grid lg:grid-cols-2 gap-12 lg:gap-8 items-center min-h-[calc(100vh-5rem)]">
          {/* Left Content */}
          <div ref={titleRef} className="flex flex-col justify-center">
            <div className="space-y-2">
              <div className="hero-title-line overflow-hidden">
                <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold text-white tracking-tight leading-none">
                  DIGITALISER
                </h1>
              </div>
              <div className="hero-title-line overflow-hidden">
                <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold text-white tracking-tight leading-none">
                  LES SERVICES
                </h1>
              </div>
              <div className="hero-title-line overflow-hidden">
                <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold text-white tracking-tight leading-none">
                  OPTIMISER LES
                </h1>
              </div>
              <div className="hero-title-line overflow-hidden">
                <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold tracking-tight leading-none gradient-text">
                  RESSOURCES
                </h1>
              </div>
            </div>

            {/* Underline */}
            <div className="mt-6 w-32 h-1 bg-gradient-to-r from-red-600 to-orange-500" />

            {/* Subtitle */}
            <p className="hero-subtitle mt-8 text-lg md:text-xl text-white/60 max-w-lg">
              Conseil, implémentation et support pour les entreprises et les collectivités.
            </p>

            {/* CTA */}
            <div className="hero-cta mt-10">
              <button
                onClick={() => handleNavClick('contact')}
                className="btn-primary text-lg"
              >
                <span>Démarrer un projet</span>
                <ArrowRight className="w-5 h-5" />
              </button>
            </div>
          </div>

          {/* Right Content - Image */}
          <div ref={imageRef} className="hidden lg:flex justify-center items-center">
            <div className="relative">
              {/* Circle frame */}
              <div className="w-[500px] h-[500px] xl:w-[550px] xl:h-[550px] rounded-full overflow-hidden border-2 border-white/10 relative">
                <img
                  src="https://images.unsplash.com/photo-1522071820081-009f0129c71c?w=800&q=80"
                  alt="ZLO Technologies Team"
                  className="w-full h-full object-cover"
                />
                {/* Overlay gradient */}
                <div className="absolute inset-0 bg-gradient-to-t from-[#0a0a0a]/30 to-transparent" />
              </div>

              {/* Decorative ring */}
              <div className="absolute -inset-4 border border-white/5 rounded-full" />
              <div className="absolute -inset-8 border border-white/5 rounded-full" />
            </div>
          </div>
        </div>
      </div>

      {/* Scroll Indicator */}
      <div className="absolute bottom-8 left-1/2 -translate-x-1/2 flex flex-col items-center gap-2 animate-scroll-indicator">
        <span className="text-xs text-white/40 uppercase tracking-widest">Scroll</span>
        <ChevronDown className="w-5 h-5 text-white/40" />
      </div>
    </section>
  );
}
