// SectionsManager component
import { Eye, EyeOff, Menu, GripVertical, RefreshCw } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Switch } from '@/components/ui/switch';
import { Label } from '@/components/ui/label';
import { useSections } from '@/context/SectionsContext';
import { useToast } from '@/hooks/use-toast';

export default function SectionsManager() {
  const { 
    sections, 
    menuItems, 
    updateSection, 
    toggleSectionVisibility, 
    toggleMenuItemVisibility,
    resetToDefaults 
  } = useSections();
  const { toast } = useToast();

  const handleReset = () => {
    if (confirm('Êtes-vous sûr de vouloir réinitialiser la configuration des sections ?')) {
      resetToDefaults();
      toast({
        title: 'Réinitialisation',
        description: 'La configuration a été réinitialisée.',
      });
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-white">Gestion des sections</h1>
          <p className="text-white/60">Contrôlez la visibilité et l'ordre des sections</p>
        </div>
        <Button variant="outline" onClick={handleReset} className="border-red-500/30 text-red-400 hover:bg-red-500/10">
          <RefreshCw className="w-4 h-4 mr-2" />
          Réinitialiser
        </Button>
      </div>

      {/* Sections */}
      <div className="space-y-4">
        <h2 className="text-lg font-semibold text-white">Sections du site</h2>
        <div className="space-y-2">
          {sections
            .sort((a, b) => a.order - b.order)
            .map((section) => (
            <div
              key={section.id}
              className="flex items-center justify-between p-4 bg-[#141419] border border-white/10 rounded-lg"
            >
              <div className="flex items-center gap-4">
                <GripVertical className="w-5 h-5 text-white/30 cursor-move" />
                <div>
                  <h3 className="text-white font-medium">{section.name}</h3>
                  <p className="text-white/50 text-sm">{section.description}</p>
                </div>
              </div>
              
              <div className="flex items-center gap-6">
                <div className="flex items-center gap-2">
                  <Switch
                    checked={section.isVisible}
                    onCheckedChange={() => toggleSectionVisibility(section.id)}
                  />
                  <Label className="text-white/60 text-sm">
                    {section.isVisible ? <Eye className="w-4 h-4" /> : <EyeOff className="w-4 h-4" />}
                  </Label>
                </div>
                
                <div className="flex items-center gap-2">
                  <Switch
                    checked={section.showInMenu}
                    onCheckedChange={() => updateSection(section.id, { showInMenu: !section.showInMenu })}
                  />
                  <Label className="text-white/60 text-sm flex items-center gap-1">
                    <Menu className="w-4 h-4" />
                    Menu
                  </Label>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Menu Items */}
      <div className="space-y-4">
        <h2 className="text-lg font-semibold text-white">Éléments du menu</h2>
        <div className="space-y-2">
          {menuItems
            .sort((a, b) => a.order - b.order)
            .map((item) => (
            <div
              key={item.id}
              className="flex items-center justify-between p-4 bg-[#141419] border border-white/10 rounded-lg"
            >
              <div className="flex items-center gap-4">
                <GripVertical className="w-5 h-5 text-white/30 cursor-move" />
                <div>
                  <h3 className="text-white font-medium">{item.label}</h3>
                  <p className="text-white/50 text-sm">{item.href}</p>
                </div>
              </div>
              
              <div className="flex items-center gap-2">
                <Switch
                  checked={item.isVisible}
                  onCheckedChange={() => toggleMenuItemVisibility(item.id)}
                />
                <Label className="text-white/60 text-sm">
                  {item.isVisible ? <Eye className="w-4 h-4" /> : <EyeOff className="w-4 h-4" />}
                </Label>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Info */}
      <div className="p-4 bg-blue-500/10 border border-blue-500/20 rounded-lg">
        <p className="text-blue-400 text-sm">
          <strong>Info:</strong> Les modifications sont automatiquement sauvegardées. 
          Les sections masquées ne seront pas affichées sur le site, mais leur contenu est conservé.
        </p>
      </div>
    </div>
  );
}
