import { useState } from 'react';
import { Save, RefreshCw } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { useToast } from '@/hooks/use-toast';

interface ThemeColors {
  primary: string;
  secondary: string;
  accent: string;
}

const defaultColors: ThemeColors = {
  primary: '#dc2626',
  secondary: '#f97316',
  accent: '#ea580c'
};

export default function ThemeSettings() {
  const { toast } = useToast();
  const [colors, setColors] = useState<ThemeColors>(() => {
    const saved = localStorage.getItem('zlo_theme_colors');
    return saved ? JSON.parse(saved) : defaultColors;
  });

  const handleSave = () => {
    localStorage.setItem('zlo_theme_colors', JSON.stringify(colors));
    toast({
      title: 'Thème sauvegardé',
      description: 'Les couleurs du thème ont été mises à jour.',
    });
  };

  const handleReset = () => {
    setColors(defaultColors);
    localStorage.setItem('zlo_theme_colors', JSON.stringify(defaultColors));
    toast({
      title: 'Thème réinitialisé',
      description: 'Les couleurs par défaut ont été restaurées.',
    });
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-white">Paramètres du thème</h1>
          <p className="text-white/60">Personnalisez les couleurs de votre site</p>
        </div>
        <div className="flex gap-2">
          <Button variant="outline" onClick={handleReset} className="border-white/10 text-white hover:bg-white/10">
            <RefreshCw className="w-4 h-4 mr-2" />
            Réinitialiser
          </Button>
          <Button onClick={handleSave} className="bg-red-500 hover:bg-red-600">
            <Save className="w-4 h-4 mr-2" />
            Sauvegarder
          </Button>
        </div>
      </div>

      {/* Color Settings */}
      <div className="p-6 bg-[#141419] border border-white/10 rounded-lg space-y-6">
        <div className="grid sm:grid-cols-3 gap-6">
          {/* Primary Color */}
          <div>
            <Label className="text-white mb-2 block">Couleur principale</Label>
            <div className="flex gap-2">
              <input
                type="color"
                value={colors.primary}
                onChange={(e) => setColors(prev => ({ ...prev, primary: e.target.value }))}
                className="w-12 h-10 rounded cursor-pointer"
              />
              <Input
                value={colors.primary}
                onChange={(e) => setColors(prev => ({ ...prev, primary: e.target.value }))}
                className="flex-1 bg-white/5 border-white/10 text-white"
              />
            </div>
            <p className="text-white/40 text-sm mt-1">Utilisée pour les boutons principaux et liens</p>
          </div>

          {/* Secondary Color */}
          <div>
            <Label className="text-white mb-2 block">Couleur secondaire</Label>
            <div className="flex gap-2">
              <input
                type="color"
                value={colors.secondary}
                onChange={(e) => setColors(prev => ({ ...prev, secondary: e.target.value }))}
                className="w-12 h-10 rounded cursor-pointer"
              />
              <Input
                value={colors.secondary}
                onChange={(e) => setColors(prev => ({ ...prev, secondary: e.target.value }))}
                className="flex-1 bg-white/5 border-white/10 text-white"
              />
            </div>
            <p className="text-white/40 text-sm mt-1">Utilisée pour les accents et dégradés</p>
          </div>

          {/* Accent Color */}
          <div>
            <Label className="text-white mb-2 block">Couleur d'accent</Label>
            <div className="flex gap-2">
              <input
                type="color"
                value={colors.accent}
                onChange={(e) => setColors(prev => ({ ...prev, accent: e.target.value }))}
                className="w-12 h-10 rounded cursor-pointer"
              />
              <Input
                value={colors.accent}
                onChange={(e) => setColors(prev => ({ ...prev, accent: e.target.value }))}
                className="flex-1 bg-white/5 border-white/10 text-white"
              />
            </div>
            <p className="text-white/40 text-sm mt-1">Utilisée pour les badges et éléments spéciaux</p>
          </div>
        </div>

        {/* Preview */}
        <div className="mt-8">
          <Label className="text-white mb-4 block">Aperçu</Label>
          <div className="p-6 rounded-lg bg-[#0a0a0f] space-y-4">
            <div className="flex gap-4">
              <button
                className="px-6 py-3 rounded-lg text-white font-medium transition-colors"
                style={{ backgroundColor: colors.primary }}
              >
                Bouton principal
              </button>
              <button
                className="px-6 py-3 rounded-lg text-white font-medium transition-colors"
                style={{ backgroundColor: colors.secondary }}
              >
                Bouton secondaire
              </button>
              <span
                className="px-4 py-2 rounded-full text-white text-sm font-medium"
                style={{ backgroundColor: colors.accent }}
              >
                Badge
              </span>
            </div>
            <div 
              className="h-2 rounded-full"
              style={{ 
                background: `linear-gradient(to right, ${colors.primary}, ${colors.secondary})` 
              }}
            />
          </div>
        </div>
      </div>

      {/* Info */}
      <div className="p-4 bg-blue-500/10 border border-blue-500/20 rounded-lg">
        <p className="text-blue-400 text-sm">
          <strong>Info:</strong> Les modifications de thème s'appliquent au site principal. 
          Le sous-site FBO a son propre système de thème accessible depuis la section FBO.
        </p>
      </div>
    </div>
  );
}
