import { useState } from 'react';
import { useParams, Link } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { ArrowLeft, ShoppingCart, Check, Minus, Plus, Leaf } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { useFBOAdmin } from '@/context/FBOAdminContext';
import { usePriceFormatter } from '@/context/CurrencyContext';

export default function FBOProductDetail() {
  const { t, i18n } = useTranslation();
  const { id } = useParams<{ id: string }>();
  const { products, addToCart, theme } = useFBOAdmin();
  const formatPrice = usePriceFormatter();
  const [quantity, setQuantity] = useState(1);

  const product = products.find(p => p.id === id);

  if (!product) {
    return (
      <div className="container mx-auto px-4 py-24 text-center">
        <p className="text-gray-500">Produit non trouvé</p>
        <Link to="/fbo/products">
          <Button className="mt-4" style={{ backgroundColor: theme.primaryColor }}>
            <ArrowLeft className="mr-2 w-4 h-4" />
            Retour aux produits
          </Button>
        </Link>
      </div>
    );
  }

  const name = i18n.language === 'fr' ? product.name : product.nameEn;
  const description = i18n.language === 'fr' ? product.description : product.descriptionEn;
  const composition = i18n.language === 'fr' ? product.composition : product.compositionEn;
  const dosage = i18n.language === 'fr' ? product.dosage : product.dosageEn;
  const benefits = i18n.language === 'fr' ? product.benefits : product.benefitsEn;

  const handleAddToCart = () => {
    addToCart(product, quantity);
  };

  return (
    <div className="min-h-screen" style={{ backgroundColor: theme.backgroundColor }}>
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Breadcrumb */}
        <nav className="flex items-center gap-2 text-sm mb-8">
          <Link to="/fbo" className="text-gray-500 hover:text-[#2D5A27]">Accueil</Link>
          <span className="text-gray-300">/</span>
          <Link to="/fbo/products" className="text-gray-500 hover:text-[#2D5A27]">Produits</Link>
          <span className="text-gray-300">/</span>
          <span style={{ color: theme.textColor }}>{name}</span>
        </nav>

        <div className="grid lg:grid-cols-2 gap-12">
          {/* Image */}
          <div className="rounded-2xl overflow-hidden bg-white shadow-sm">
            <img
              src={product.image}
              alt={name}
              className="w-full aspect-square object-cover"
            />
          </div>

          {/* Info */}
          <div>
            <div className="flex items-center gap-3 mb-4">
              <Badge style={{ backgroundColor: `${theme.primaryColor}20`, color: theme.primaryColor }}>
                {t(`fbo.categories.${product.category}`)}
              </Badge>
              {product.featured && (
                <Badge style={{ backgroundColor: theme.accentColor, color: 'white' }}>
                  Vedette
                </Badge>
              )}
              {!product.inStock && (
                <Badge variant="destructive">{t('fbo.outOfStock')}</Badge>
              )}
            </div>

            <h1 
              className="text-3xl sm:text-4xl font-bold mb-4"
              style={{ color: theme.textColor }}
            >
              {name}
            </h1>

            <p 
              className="text-sm mb-4"
              style={{ color: `${theme.textColor}60` }}
            >
              Réf: {product.reference}
            </p>

            <p 
              className="text-lg mb-6"
              style={{ color: `${theme.textColor}90` }}
            >
              {description}
            </p>

            <div 
              className="text-3xl font-bold mb-8"
              style={{ color: theme.primaryColor }}
            >
              {formatPrice(product.price)}
            </div>

            {/* Quantity & Add to Cart */}
            {product.inStock && (
              <div className="flex items-center gap-4 mb-8">
                <div className="flex items-center border rounded-lg">
                  <button
                    onClick={() => setQuantity(Math.max(1, quantity - 1))}
                    className="p-3 hover:bg-gray-100 transition-colors"
                  >
                    <Minus className="w-4 h-4" />
                  </button>
                  <span className="w-12 text-center font-medium">{quantity}</span>
                  <button
                    onClick={() => setQuantity(quantity + 1)}
                    className="p-3 hover:bg-gray-100 transition-colors"
                  >
                    <Plus className="w-4 h-4" />
                  </button>
                </div>

                <Button
                  size="lg"
                  onClick={handleAddToCart}
                  className="flex-1 text-white"
                  style={{ backgroundColor: theme.primaryColor }}
                >
                  <ShoppingCart className="w-5 h-5 mr-2" />
                  {t('fbo.addToCart')}
                </Button>
              </div>
            )}

            {/* Stock status */}
            {product.inStock ? (
              <div className="flex items-center gap-2 text-green-600">
                <Check className="w-5 h-5" />
                <span>{t('fbo.inStock')}</span>
              </div>
            ) : (
              <div className="flex items-center gap-2 text-red-500">
                <span>{t('fbo.outOfStock')}</span>
              </div>
            )}
          </div>
        </div>

        {/* Tabs */}
        <div className="mt-16">
          <Tabs defaultValue="composition">
            <TabsList className="w-full justify-start border-b rounded-none bg-transparent">
              <TabsTrigger 
                value="composition"
                className="data-[state=active]:border-b-2 data-[state=active]:border-[#2D5A27] rounded-none"
              >
                {t('fbo.composition')}
              </TabsTrigger>
              <TabsTrigger 
                value="dosage"
                className="data-[state=active]:border-b-2 data-[state=active]:border-[#2D5A27] rounded-none"
              >
                {t('fbo.dosage')}
              </TabsTrigger>
              <TabsTrigger 
                value="benefits"
                className="data-[state=active]:border-b-2 data-[state=active]:border-[#2D5A27] rounded-none"
              >
                {t('fbo.benefits')}
              </TabsTrigger>
            </TabsList>

            <TabsContent value="composition" className="mt-6">
              <div 
                className="p-6 rounded-xl"
                style={{ backgroundColor: theme.cardBackground }}
              >
                <p style={{ color: theme.textColor }}>{composition}</p>
              </div>
            </TabsContent>

            <TabsContent value="dosage" className="mt-6">
              <div 
                className="p-6 rounded-xl"
                style={{ backgroundColor: theme.cardBackground }}
              >
                <p style={{ color: theme.textColor }}>{dosage}</p>
              </div>
            </TabsContent>

            <TabsContent value="benefits" className="mt-6">
              <div 
                className="p-6 rounded-xl"
                style={{ backgroundColor: theme.cardBackground }}
              >
                <ul className="space-y-3">
                  {benefits.map((benefit, index) => (
                    <li key={index} className="flex items-start gap-3">
                      <Leaf 
                        className="w-5 h-5 mt-0.5 flex-shrink-0"
                        style={{ color: theme.primaryColor }}
                      />
                      <span style={{ color: theme.textColor }}>{benefit}</span>
                    </li>
                  ))}
                </ul>
              </div>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </div>
  );
}
