import { Link } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { ArrowLeft, Trash2, Minus, Plus, ShoppingBag, Phone, MessageCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useFBOAdmin } from '@/context/FBOAdminContext';
import { usePriceFormatter } from '@/context/CurrencyContext';

export default function FBOCart() {
  const { t, i18n } = useTranslation();
  const { cart, removeFromCart, updateCartQuantity, getCartTotal, clearCart, theme, contact } = useFBOAdmin();
  const formatPrice = usePriceFormatter();

  if (cart.length === 0) {
    return (
      <div 
        className="min-h-screen flex items-center justify-center"
        style={{ backgroundColor: theme.backgroundColor }}
      >
        <div className="text-center px-4">
          <div 
            className="w-24 h-24 rounded-full flex items-center justify-center mx-auto mb-6"
            style={{ backgroundColor: `${theme.primaryColor}15` }}
          >
            <ShoppingBag className="w-12 h-12" style={{ color: theme.primaryColor }} />
          </div>
          <h2 
            className="text-2xl font-bold mb-4"
            style={{ color: theme.textColor }}
          >
            {t('fbo.emptyCart')}
          </h2>
          <p 
            className="mb-8"
            style={{ color: `${theme.textColor}70` }}
          >
            Découvrez nos produits et ajoutez-les à votre panier
          </p>
          <Link to="/fbo/products">
            <Button 
              size="lg"
              style={{ backgroundColor: theme.primaryColor }}
              className="text-white"
            >
              <ArrowLeft className="mr-2 w-4 h-4" />
              {t('fbo.continueShopping')}
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div 
      className="min-h-screen py-8"
      style={{ backgroundColor: theme.backgroundColor }}
    >
      <div className="container mx-auto px-4 sm:px-6 lg:px-8">
        <h1 
          className="text-3xl font-bold mb-8"
          style={{ color: theme.textColor }}
        >
          {t('fbo.cart')}
        </h1>

        <div className="grid lg:grid-cols-3 gap-8">
          {/* Cart Items */}
          <div className="lg:col-span-2 space-y-4">
            {cart.map((item) => (
              <div
                key={item.product.id}
                className="flex gap-4 p-4 rounded-xl"
                style={{ backgroundColor: theme.cardBackground }}
              >
                <img
                  src={item.product.image}
                  alt={i18n.language === 'fr' ? item.product.name : item.product.nameEn}
                  className="w-24 h-24 object-cover rounded-lg"
                />
                
                <div className="flex-1">
                  <h3 
                    className="font-semibold mb-1"
                    style={{ color: theme.textColor }}
                  >
                    {i18n.language === 'fr' ? item.product.name : item.product.nameEn}
                  </h3>
                  <p 
                    className="text-sm mb-2"
                    style={{ color: `${theme.textColor}60` }}
                  >
                    Réf: {item.product.reference}
                  </p>
                  <p 
                    className="font-medium"
                    style={{ color: theme.primaryColor }}
                  >
                    {formatPrice(item.product.price)}
                  </p>
                </div>

                <div className="flex flex-col items-end justify-between">
                  <button
                    onClick={() => removeFromCart(item.product.id)}
                    className="p-2 text-red-500 hover:bg-red-50 rounded-lg transition-colors"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>

                  <div className="flex items-center border rounded-lg">
                    <button
                      onClick={() => updateCartQuantity(item.product.id, item.quantity - 1)}
                      className="p-2 hover:bg-gray-100 transition-colors"
                    >
                      <Minus className="w-4 h-4" />
                    </button>
                    <span className="w-10 text-center font-medium">{item.quantity}</span>
                    <button
                      onClick={() => updateCartQuantity(item.product.id, item.quantity + 1)}
                      className="p-2 hover:bg-gray-100 transition-colors"
                    >
                      <Plus className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              </div>
            ))}

            <Button
              variant="outline"
              onClick={clearCart}
              className="w-full"
            >
              <Trash2 className="mr-2 w-4 h-4" />
              Vider le panier
            </Button>
          </div>

          {/* Summary */}
          <div className="space-y-4">
            <div 
              className="p-6 rounded-xl"
              style={{ backgroundColor: theme.cardBackground }}
            >
              <h3 
                className="text-lg font-semibold mb-4"
                style={{ color: theme.textColor }}
              >
                Récapitulatif
              </h3>
              
              <div className="space-y-2 mb-4">
                <div className="flex justify-between">
                  <span style={{ color: `${theme.textColor}70` }}>Sous-total</span>
                  <span style={{ color: theme.textColor }}>{formatPrice(getCartTotal())}</span>
                </div>
                <div className="flex justify-between">
                  <span style={{ color: `${theme.textColor}70` }}>Livraison</span>
                  <span style={{ color: theme.textColor }}>À calculer</span>
                </div>
              </div>

              <div 
                className="border-t pt-4 mb-6"
                style={{ borderColor: `${theme.textColor}20` }}
              >
                <div className="flex justify-between">
                  <span 
                    className="font-semibold"
                    style={{ color: theme.textColor }}
                  >
                    {t('fbo.total')}
                  </span>
                  <span 
                    className="text-xl font-bold"
                    style={{ color: theme.primaryColor }}
                  >
                    {formatPrice(getCartTotal())}
                  </span>
                </div>
              </div>

              <p 
                className="text-sm mb-4"
                style={{ color: `${theme.textColor}60` }}
              >
                Pour finaliser votre commande, contactez-nous directement :
              </p>

              <div className="space-y-3">
                <a href={`tel:${contact.phone}`}>
                  <Button 
                    className="w-full text-white"
                    style={{ backgroundColor: theme.primaryColor }}
                  >
                    <Phone className="mr-2 w-4 h-4" />
                    {t('fbo.orderByPhone')}
                  </Button>
                </a>
                
                <a 
                  href={`https://wa.me/${contact.whatsapp?.replace(/\D/g, '')}`}
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  <Button 
                    variant="outline" 
                    className="w-full"
                    style={{ borderColor: '#25D366', color: '#25D366' }}
                  >
                    <MessageCircle className="mr-2 w-4 h-4" />
                    {t('fbo.orderByWhatsApp')}
                  </Button>
                </a>
              </div>
            </div>

            <Link to="/fbo/products">
              <Button variant="ghost" className="w-full">
                <ArrowLeft className="mr-2 w-4 h-4" />
                {t('fbo.continueShopping')}
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
}
