import { useEffect, useRef } from 'react';
import { Link } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { ArrowRight, Leaf, Users } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useFBOAdmin } from '@/context/FBOAdminContext';
import gsap from 'gsap';

export default function FBOHero() {
  const { i18n } = useTranslation();
  const { hero, theme } = useFBOAdmin();
  const heroRef = useRef<HTMLElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      gsap.fromTo(contentRef.current?.children || [],
        { opacity: 0, y: 30 },
        {
          opacity: 1,
          y: 0,
          duration: 0.8,
          stagger: 0.15,
          ease: 'power3.out',
          delay: 0.3
        }
      );
    }, heroRef);

    return () => ctx.revert();
  }, []);

  const title = i18n.language === 'fr' ? hero.title : hero.titleEn;
  const subtitle = i18n.language === 'fr' ? hero.subtitle : hero.subtitleEn;
  const description = i18n.language === 'fr' ? hero.description : hero.descriptionEn;
  const ctaPrimary = i18n.language === 'fr' ? hero.ctaPrimary : hero.ctaPrimaryEn;
  const ctaSecondary = i18n.language === 'fr' ? hero.ctaSecondary : hero.ctaSecondaryEn;

  return (
    <section
      ref={heroRef}
      className="relative min-h-[80vh] flex items-center justify-center overflow-hidden pt-20"
      style={{ backgroundColor: theme.backgroundColor }}
    >
      {/* Background Image */}
      <div 
        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
        style={{ backgroundImage: `url(${hero.backgroundImage})` }}
      >
        <div 
          className="absolute inset-0"
          style={{ 
            background: `linear-gradient(to bottom, ${theme.backgroundColor}ee, ${theme.backgroundColor}99, ${theme.backgroundColor})` 
          }}
        />
      </div>

      {/* Decorative elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div 
          className="absolute top-1/4 left-1/4 w-64 h-64 rounded-full blur-3xl opacity-20"
          style={{ backgroundColor: theme.primaryColor }}
        />
        <div 
          className="absolute bottom-1/4 right-1/4 w-64 h-64 rounded-full blur-3xl opacity-20"
          style={{ backgroundColor: theme.accentColor }}
        />
      </div>

      {/* Content */}
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        <div ref={contentRef} className="max-w-3xl mx-auto text-center">
          {/* Badge */}
          <div 
            className="inline-flex items-center gap-2 px-4 py-2 rounded-full text-sm font-medium mb-6"
            style={{ 
              backgroundColor: `${theme.primaryColor}20`,
              color: theme.primaryColor 
            }}
          >
            <Leaf className="w-4 h-4" />
            <span>Forever Living Products</span>
          </div>

          <h1 
            className="text-4xl sm:text-5xl md:text-6xl font-bold mb-4"
            style={{ color: theme.textColor }}
          >
            {title}
          </h1>
          
          <p 
            className="text-xl sm:text-2xl font-semibold mb-6"
            style={{ color: theme.primaryColor }}
          >
            {subtitle}
          </p>
          
          <p 
            className="text-lg mb-8 max-w-2xl mx-auto"
            style={{ color: `${theme.textColor}99` }}
          >
            {description}
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/fbo/products">
              <Button
                size="lg"
                className="text-white font-semibold px-8 py-6 text-lg group"
                style={{ backgroundColor: theme.primaryColor }}
              >
                {ctaPrimary}
                <ArrowRight className="ml-2 w-5 h-5 group-hover:translate-x-1 transition-transform" />
              </Button>
            </Link>
            
            <a 
              href="https://www.foreverliving.com" 
              target="_blank" 
              rel="noopener noreferrer"
            >
              <Button
                size="lg"
                variant="outline"
                className="px-8 py-6 text-lg"
                style={{ 
                  borderColor: theme.primaryColor,
                  color: theme.primaryColor 
                }}
              >
                <Users className="mr-2 w-5 h-5" />
                {ctaSecondary}
              </Button>
            </a>
          </div>
        </div>
      </div>
    </section>
  );
}
